unit FMXTee.Bind.Editors;

interface

implementation

uses
  System.Classes, System.SysUtils, FMXTee.Procs, FMXTee.Engine, FMXTee.Chart,
  Data.Bind.Components, Data.Bind.Editors, System.Bindings.EvalProtocol,
  System.Bindings.ObjEval, FMXTee.Series;

type
  TSeriesEditorObject=class
  private
    FIndex: Integer;
    FItemIndex: Integer;
    FSeries: TChartSeries;
  end;

  TBindListSeriesEditor = class(TBindListEditor)
  private
    FEditorObject: TSeriesEditorObject;
  public
    constructor Create(ASeries: TChartSeries);
    destructor Destroy; override;
    procedure BeginUpdate; override;
    procedure EndUpdate; override;
    procedure DeleteToEnd; override;
    function AddItem(Select: Boolean): IScope; override;
    function InsertItem(Select: Boolean): IScope; override;
    function CanInsertItem: Boolean; override;
    function MoveNext: Boolean; override;
    function GetRowCount: Integer; override;
    function CurrentItem: IScope; override;
    procedure ClearList; override;
    function GetSelectedText: string; override;
    procedure SetSelectedText(const AValue: string); override;
  end;

  TBindSeriesEditorFactory = class(TBindEditorFactory)
  public
    constructor Create; override;
    function Supports(AIntf: TGuid; AObject: TObject): Boolean; override;
    function CreateEditor(AIntf: TGuid;
      AObject: TObject): IInterface; override;
  end;

  TChartItemEditorObject = class
  private
    FChart: TCustomChart;
    FIndex: Integer;
    FItemIndex: Integer;
    function GetOwner: TCustomChart;
    function GetValues(ACol: Integer): string;
    procedure SetValues(ACol: Integer; const Value: string);
  public
    property Owner: TCustomChart read GetOwner;
    property Values[ACol: Integer]: string read GetValues write SetValues;
  end;

  TBindListChartEditor = class(TBindGridEditor)
  private
    FEditorObject: TChartItemEditorObject;
    function IsEmpty: Boolean;
    function GetRowIndex: Integer;
    function GetColumnIndex: Integer;
  protected
    procedure BeginUpdate; override;
    procedure EndUpdate; override;
    function AddItem(Select: Boolean): IScope; override;
    function InsertItem(Select: Boolean): IScope; override;
    function CanInsertItem: Boolean; override;
    function CurrentItem: IScope; override;
    function GetRowCount: Integer; override;
    function MoveNext: Boolean; override;
    procedure DeleteToEnd; override;
    procedure ClearList; override;
    function GetSelectedText: string; override;
    procedure SetSelectedText(const AValue: string); override;
    procedure GetColumnNames(ANames: TStrings); override;
    procedure GetColumnIndices(ANames: TStrings); override;
  public
    constructor Create(AChart: TCustomChart);
    destructor Destroy; override;
  end;

  TBindChartEditorFactory = class(TBindEditorFactory)
  public
    constructor Create; override;
    function Supports(AIntf: TGuid; AObject: TObject): Boolean; override;
    function CreateEditor(AIntf: TGuid;
      AObject: TObject): IInterface; override;
  end;

function SeriesValue(ASeries:TChartSeries; AIndex:Integer):String;
var tmpValue : TChartValue;
begin
  tmpValue:=ASeries.MandatoryValueList[AIndex];

  if ASeries.MandatoryValueList.DateTime then
     DateTimeToString(result,DateTimeDefaultFormat(tmpValue),tmpValue)
  else
     result:=FormatFloat(ASeries.ValueFormat,tmpValue);
end;

{ TBindListSeriesEditor }

function TBindListSeriesEditor.AddItem(Select: Boolean): IScope;
begin
  FEditorObject.FIndex := FEditorObject.FSeries.Add(0);
  if Select then
    FEditorObject.FItemIndex := FEditorObject.FIndex;
  Result := WrapObject(FEditorObject);
end;

function TBindListSeriesEditor.MoveNext: Boolean;
begin
  if FEditorObject.FSeries.Count = 0 then
    Exit(False);
  if FEditorObject.FIndex = -1 then
  begin
    FEditorObject.FIndex := FEditorObject.FItemIndex;
    if FEditorObject.FIndex < 0 then
      FEditorObject.FIndex := 0;
  end
  else
    FEditorObject.FIndex := FEditorObject.FIndex + 1;
  Result := (FEditorObject.FIndex >= 0) and (FEditorObject.FIndex < FEditorObject.FSeries.Count);
end;

function TBindListSeriesEditor.GetRowCount: Integer;
begin
  Result := FEditorObject.FSeries.Count;
end;

function TBindListSeriesEditor.CurrentItem: IScope;
begin
  if FEditorObject.FIndex = -1 then
    FEditorObject.FIndex := FEditorObject.FItemIndex;

  if FEditorObject.FIndex <> -1 then
    Result := WrapObject(FEditorObject)
  else
    Result := nil;
end;

procedure TBindListSeriesEditor.BeginUpdate;
begin
  FEditorObject.FSeries.BeginUpdate;
end;

function TBindListSeriesEditor.CanInsertItem: Boolean;
begin
  Result := FEditorObject.FItemIndex <> -1;
end;

procedure TBindListSeriesEditor.ClearList;
begin
  FEditorObject.FSeries.Clear;
end;

constructor TBindListSeriesEditor.Create(ASeries: TChartSeries);
begin
  FEditorObject := TSeriesEditorObject.Create;
  FEditorObject.FIndex := -1;
  FEditorObject.FSeries := ASeries;
  FEditorObject.FItemIndex:= -1;
end;

destructor TBindListSeriesEditor.Destroy;
begin
  FEditorObject.Free;
  inherited;
end;

procedure TBindListSeriesEditor.EndUpdate;
begin
  FEditorObject.FSeries.EndUpdate;
end;

function TBindListSeriesEditor.GetSelectedText: string;
begin
  Result := '';

  with FEditorObject do
  if FItemIndex <> -1 then
     if FItemIndex < FSeries.Count then
        result:=SeriesValue(FSeries,FItemIndex);
end;

function TBindListSeriesEditor.InsertItem(Select: Boolean): IScope;
begin
  if not CanInsertItem then
    Result := nil
  else
  begin
    FEditorObject.FIndex := FEditorObject.FItemIndex;
    FEditorObject.FSeries.AddXY(FEditorObject.FIndex, 0);

    if Select then
      FEditorObject.FItemIndex := FEditorObject.FIndex;

    Result := WrapObject(FEditorObject);
  end;
end;

procedure TBindListSeriesEditor.SetSelectedText(const AValue: string);
var
  t: Integer;
begin
  with FEditorObject,FSeries do
  for t := 0 to Count - 1 do
      if SameText(SeriesValue(FSeries,t), AValue) then
      begin
        FEditorObject.FItemIndex := t;
        Exit;
      end;

  FEditorObject.FItemIndex := -1;
end;

procedure TBindListSeriesEditor.DeleteToEnd;
begin
  if FEditorObject.FIndex = -1 then
    FEditorObject.FIndex := FEditorObject.FItemIndex;

  FEditorObject.FSeries.Delete(FEditorObject.FIndex, FEditorObject.FSeries.Count-FEditorObject.FIndex);
  FEditorObject.FIndex := FEditorObject.FSeries.Count - 1;
end;

{ TBindSeriesEditorFactory }

constructor TBindSeriesEditorFactory.Create;
begin
  inherited;
end;

function TBindSeriesEditorFactory.CreateEditor(AIntf: TGuid;
  AObject: TObject): IInterface;
begin
  Result := TBindListSeriesEditor.Create(TChartSeries(AObject));
end;

function TBindSeriesEditorFactory.Supports(AIntf: TGuid; AObject:
  TObject): Boolean;
begin
  Result := (AIntf = IBindListEditor) and AObject.InheritsFrom(TChartSeries);
end;

{ TChartItemEditorObject }

function TChartItemEditorObject.GetValues(ACol: Integer): string;
begin
  Result := SeriesValue(FChart[ACol],FIndex);
end;

function TChartItemEditorObject.GetOwner: TCustomChart;
begin
  Result := FChart;
end;

procedure TChartItemEditorObject.SetValues(ACol: Integer;
  const Value: string);
begin
  if ACol>=FChart.SeriesCount then
     FChart.AddSeries(TBarSeries);

  if (FIndex > -1) and (FIndex <= FChart[ACol].Count) then
     FChart[ACol].MandatoryValueList[FIndex] := StrToFloat(Value);
end;

{ TBindChartEditorFactory }

constructor TBindChartEditorFactory.Create;
begin
  inherited;
end;

function TBindChartEditorFactory.CreateEditor(AIntf: TGuid;
  AObject: TObject): IInterface;
begin
  Result := TBindListChartEditor.Create(TCustomChart(AObject));
end;

function TBindChartEditorFactory.Supports(AIntf: TGuid; AObject:
  TObject): Boolean;
begin
  Result := (AIntf = IBindListEditor) and (AObject <> nil) and
            AObject.InheritsFrom(TCustomChart);
end;

{ TBindListChartEditor }

function TBindListChartEditor.AddItem(Select: Boolean): IScope;
var t : Integer;
begin
  for t := 0 to FEditorObject.FChart.SeriesCount-1 do
      FEditorObject.FChart[t].AddNull;

  FEditorObject.FIndex := FEditorObject.FChart[0].Count - 1;

  if Select then
    FEditorObject.FItemIndex := FEditorObject.FIndex;

  Result := WrapObject(FEditorObject);
end;

function TBindListChartEditor.GetRowIndex: Integer;
begin
  if FEditorObject.FItemIndex >= FEditorObject.FChart[0].Count then
    Result := FEditorObject.FChart[0].Count - 1
  else
    Result := FEditorObject.FItemIndex;
end;

function TBindListChartEditor.GetColumnIndex: Integer;
begin
  Result := FEditorObject.FIndex;
end;

procedure TBindListChartEditor.GetColumnIndices(ANames: TStrings);
var
  I: Integer;
begin
  for I := 0 to FEditorObject.FChart.SeriesCount- 1 do
    ANames.Add(IntToStr(I));
end;

procedure TBindListChartEditor.GetColumnNames(ANames: TStrings);
var
  I: Integer;
begin
  for I := 0 to FEditorObject.FChart.SeriesCount - 1 do
    ANames.Add(FEditorObject.FChart[I].Name);
end;

function TBindListChartEditor.CurrentItem: IScope;
begin
  //FEditorObject.FIndex := FEditorObject.FStringGrid.Row;
  if FEditorObject.FIndex = -1 then
    FEditorObject.FIndex := GetRowIndex;
  Result := WrapObject(FEditorObject);
end;

procedure TBindListChartEditor.BeginUpdate;
begin
  FEditorObject.FChart.BeginUpdate;
end;

function TBindListChartEditor.CanInsertItem: Boolean;
begin
  Result := False; // Not supported
end;

procedure TBindListChartEditor.ClearList;
begin
  FEditorObject.FIndex := -1;
  FEditorObject.FChart.SeriesList.ClearValues;
end;

constructor TBindListChartEditor.Create(AChart: TCustomChart);
begin
  FEditorObject := TChartItemEditorObject.Create;
  FEditorObject.FChart := AChart;
  FEditorObject.FIndex := -1;
  FEditorObject.FItemIndex:= -1;
end;

procedure TBindListChartEditor.DeleteToEnd;
begin
  if FEditorObject.FIndex = -1 then
    FEditorObject.FIndex := GetRowIndex;
  if FEditorObject.FIndex = -1 then
    ClearList
  else
    FEditorObject.FChart[0].Delete(FEditorObject.FIndex, FEditorObject.FChart[0].Count-FEditorObject.FIndex);
end;

destructor TBindListChartEditor.Destroy;
begin
  FEditorObject.Free;
  inherited;
end;

procedure TBindListChartEditor.EndUpdate;
begin
  FEditorObject.FChart.EndUpdate;
end;

function TBindListChartEditor.InsertItem(Select: Boolean): IScope;
begin
  Result := nil;
end;

function TBindListChartEditor.IsEmpty: Boolean;
begin
  Result := (FEditorObject.FChart.SeriesCount=0)
end;

function TBindListChartEditor.GetRowCount: Integer;
begin
  if IsEmpty then
    Result := 0
  else
    Result := FEditorObject.FChart[0].Count;
end;

function TBindListChartEditor.GetSelectedText: string;
begin
  Result := SeriesValue(FEditorObject.FChart[GetColumnIndex], GetRowIndex);
end;

function TBindListChartEditor.MoveNext: Boolean;
begin
  if FEditorObject.FChart[0].Count = 0 then
    Exit(False);
  if FEditorObject.FIndex = -1 then
    FEditorObject.FIndex :=  GetRowIndex
  else
    FEditorObject.FIndex :=  FEditorObject.FIndex + 1;
  Result := FEditorObject.FIndex < FEditorObject.FChart[0].Count;
end;

procedure TBindListChartEditor.SetSelectedText(const AValue: string);
begin
  FEditorObject.FChart[GetColumnIndex].MandatoryValueList[GetRowIndex]:=StrToFloat(AValue);
end;

initialization
  RegisterBindEditorFactory([TBindSeriesEditorFactory,
    TBindChartEditorFactory]);
finalization
  UnregisterBindEditorFactory([TBindSeriesEditorFactory,
    TBindChartEditorFactory]);
end.
